<?php

namespace App\Services;

use App\Models\Impresora;
use App\Models\PlantillaImpresion;
use App\Models\Orden;
use App\Models\Sale;
use Illuminate\Support\Facades\Log;

class PrintService
{
    /**
     * Print a Receipt (Invoice or Pre-check)
     */
    public function printOrder(Orden $orden, $tipo = 'precuenta')
    {
        $printer = Impresora::where('es_predeterminada', true)->where('activa', true)->first();
        if (!$printer) {
            Log::warning("No default printer configured.");
            return false;
        }

        $template = PlantillaImpresion::where('tipo', $tipo)->where('activa', true)->first();
        if (!$template) {
            Log::warning("No active template found for: $tipo");
            return false;
        }

        $content = $this->renderTemplate($template, $orden);
        return $this->sendToPrinter($printer, $content);
    }

    /**
     * Print a Sale (Invoice)
     */
    public function printSale(Sale $sale, $tipo = 'factura')
    {
        $printer = Impresora::where('es_predeterminada', true)->where('activa', true)->first();
        if (!$printer) {
            Log::warning("No default printer configured.");
            return false;
        }

        $template = PlantillaImpresion::where('tipo', $tipo)->where('activa', true)->first();
        if (!$template) {
            Log::warning("No active template found for: $tipo");
            return false;
        }

        $content = $this->renderTemplate($template, $sale);
        return $this->sendToPrinter($printer, $content);
    }

    /**
     * Print Arqueo (Cash Closure Report)
     */
    public function printArqueo(\App\Models\Arqueo $arqueo)
    {
        $printer = Impresora::where('es_predeterminada', true)->where('activa', true)->first();
        if (!$printer)
            return false;

        $ESC = "\x1b";
        $GS = "\x1d";
        $CENTER = $ESC . "a" . "\x01";
        $LEFT = $ESC . "a" . "\x00";
        $BOLD_ON = $ESC . "E" . "\x01";
        $BOLD_OFF = $ESC . "E" . "\x00";
        $INIT = $ESC . "@";
        $DRAWER = $ESC . "p" . "\x00" . "\x19" . "\xfa"; // Drawer kick command
        $CUT = $GS . "V" . "\x42" . "\x00";

        $config = \App\Models\BusinessConfig::first() ?: new \App\Models\BusinessConfig(['business_name' => 'SUKHA CIGAR BAR']);

        $buffer = $INIT . $DRAWER . $CENTER . $BOLD_ON;
        $buffer .= mb_strtoupper($config->business_name) . "\n";
        $buffer .= "REPORTE DE CIERRE DE CAJA\n";
        $buffer .= $BOLD_OFF;
        $buffer .= "--------------------------------\n";
        $buffer .= $LEFT;
        $buffer .= "Fecha: " . $arqueo->created_at->format('d/m/Y H:i') . "\n";
        $buffer .= "Cajero: " . $arqueo->user->name . "\n";
        $buffer .= "Caja: " . $arqueo->apertura->caja->name . "\n";
        $buffer .= "--------------------------------\n";

        $buffer .= "RESUMEN FINANCIERO\n";
        $buffer .= "Monto Inicial:  " . str_pad(number_format($arqueo->apertura->monto_inicial, 2), 16, ' ', STR_PAD_LEFT) . "\n";
        $buffer .= "Ventas Efectivo:" . str_pad(number_format($arqueo->monto_sistema - $arqueo->apertura->monto_inicial, 2), 16, ' ', STR_PAD_LEFT) . "\n";
        $buffer .= "Ventas Tarjeta: " . str_pad(number_format($arqueo->monto_tarjeta, 2), 16, ' ', STR_PAD_LEFT) . "\n";
        $buffer .= "Ventas Transf:  " . str_pad(number_format($arqueo->monto_transferencia, 2), 16, ' ', STR_PAD_LEFT) . "\n";

        // Internal Consumption (Doesn't affect cash flow)
        $internalUse = $arqueo->apertura->sales->where('is_internal', true)->sum('total');
        if ($internalUse > 0) {
            $buffer .= "Consumo Interno:" . str_pad(number_format($internalUse, 2), 16, ' ', STR_PAD_LEFT) . "\n";
        }

        $buffer .= "--------------------------------\n";

        $buffer .= $BOLD_ON . "EFECTIVO ESPERADO: " . number_format($arqueo->monto_sistema, 2) . "\n" . $BOLD_OFF;
        $buffer .= "--------------------------------\n";

        $buffer .= "DESGLOSE DE EFECTIVO\n";
        if ($arqueo->denominaciones) {
            foreach ($arqueo->denominaciones as $val => $qty) {
                if ($qty > 0) {
                    $line = str_pad("$" . $val . " x " . $qty, 20, ' ', STR_PAD_RIGHT);
                    $sub = str_pad(number_format($val * $qty, 0), 12, ' ', STR_PAD_LEFT);
                    $buffer .= "{$line}{$sub}\n";
                }
            }
        }
        $buffer .= "--------------------------------\n";

        $buffer .= $BOLD_ON . "EFECTIVO CONTADO:  " . number_format($arqueo->monto_fisico, 2) . "\n" . $BOLD_OFF;
        $buffer .= "DIFERENCIA:       " . str_pad(number_format($arqueo->diferencia, 2), 14, ' ', STR_PAD_LEFT) . "\n";

        if ($arqueo->observaciones) {
            $buffer .= "\nOBSERVACIONES:\n" . $arqueo->observaciones . "\n";
        }

        $buffer .= "\n\nFirma Cajero: _________________\n";
        $buffer .= "\n\n\n" . $CUT;

        return $this->sendToPrinter($printer, $buffer);
    }

    /**
     * Render common thermal receipt format (ESC/POS placeholder)
     */
    private function renderTemplate(PlantillaImpresion $template, $data)
    {
        $struct = $template->estructura_json;
        if (!isset($struct['bloques']) || empty($struct['bloques'])) {
            // Minimal Fallback Template
            $struct = [
                'bloques' => [
                    ['type' => 'header', 'active' => true, 'align' => 'center'],
                    ['type' => 'divider', 'active' => true],
                    ['type' => 'data_field', 'field' => 'id', 'label' => 'ORDEN: #', 'active' => true],
                    ['type' => 'data_field', 'field' => 'mesa', 'label' => 'MESA: ', 'active' => true],
                    ['type' => 'data_field', 'field' => 'fecha', 'label' => 'FECHA: ', 'active' => true],
                    ['type' => 'divider', 'active' => true],
                    ['type' => 'items_table', 'active' => true],
                    ['type' => 'divider', 'active' => true],
                    ['type' => 'totals', 'active' => true],
                ]
            ];
        }

        $buffer = "";

        // Raw ESC/POS Commands
        $ESC = "\x1b";
        $GS = "\x1d";
        $CENTER = $ESC . "a" . "\x01";
        $LEFT = $ESC . "a" . "\x00";
        $RIGHT = $ESC . "a" . "\x02";
        $BOLD_ON = $ESC . "E" . "\x01";
        $BOLD_OFF = $ESC . "E" . "\x00";
        $INIT = $ESC . "@";
        $DRAWER = $ESC . "p" . "\x00" . "\x19" . "\xfa"; // Drawer kick command
        $CUT = $GS . "V" . "\x42" . "\x00";

        $buffer .= $INIT . $DRAWER;

        $bloques = $struct['bloques'] ?? [];

        foreach ($bloques as $block) {
            if (!($block['active'] ?? true))
                continue;

            // Alignment
            $align = $block['align'] ?? 'left';
            if ($align == 'center')
                $buffer .= $CENTER;
            elseif ($align == 'right')
                $buffer .= $RIGHT;
            else
                $buffer .= $LEFT;

            switch ($block['type']) {
                case 'header':
                    $config = \App\Models\BusinessConfig::first() ?: new \App\Models\BusinessConfig([
                        'business_name' => 'SUKHA CIGAR BAR',
                        'rnc' => '123-45678-9',
                        'phone' => '(809) 555-0199',
                        'address' => 'Av. Principal #123'
                    ]);

                    if (isset($data->is_internal) && $data->is_internal) {
                        $buffer .= $BOLD_ON . "USO INTERNO - DUEÑO\n" . $BOLD_OFF;
                    }
                    $buffer .= $BOLD_ON . mb_strtoupper($config->business_name) . "\n" . $BOLD_OFF;
                    if ($config->address)
                        $buffer .= $config->address . "\n";
                    if ($config->phone)
                        $buffer .= "Tel: " . $config->phone . "\n";
                    if ($config->rnc)
                        $buffer .= "RNC: " . $config->rnc . "\n";
                    break;

                case 'text':
                    if ($block['bold'] ?? false)
                        $buffer .= $BOLD_ON;

                    $content = $block['content'] ?? '';
                    $config = \App\Models\BusinessConfig::first();
                    // Replace basic variables in custom text
                    $content = str_replace(
                        [
                            '{MESA}',
                            '{ORDEN}',
                            '{FECHA}',
                            '{HORA}',
                            '{MESERO}',
                            '{TOTAL}',
                            '{CLIENTE}',
                            '{BUSINESS_NAME}',
                            '{BUSINESS_RNC}',
                            '{BUSINESS_PHONE}',
                            '{BUSINESS_ADDRESS}'
                        ],
                        [
                            $data->mesa ? $data->mesa->name : 'N/A',
                            $data->id,
                            now()->format('d/m/Y'),
                            now()->format('H:i'),
                            $data->user ? $data->user->name : 'Admin',
                            number_format($data->total, 2),
                            $data->cliente ?? 'Final',
                            $config ? $config->business_name : 'N/A',
                            $config ? $config->rnc : 'N/A',
                            $config ? $config->phone : 'N/A',
                            $config ? $config->address : 'N/A',
                        ],
                        $content
                    );

                    $buffer .= $content . "\n";
                    if ($block['bold'] ?? false)
                        $buffer .= $BOLD_OFF;
                    break;

                case 'divider':
                    $char = ($block['style'] ?? 'dashed') == 'double' ? '=' : '-';
                    $buffer .= str_repeat($char, 32) . "\n";
                    break;

                case 'data_field':
                    $label = $block['label'] ?? '';
                    $val = "";
                    switch ($block['field'] ?? '') {
                        case 'ncf':
                            $val = $data->ncf ?? 'B0100000001';
                            break;
                        case 'id':
                        case 'orden':
                            $val = "#" . $data->id;
                            break;
                        case 'fecha':
                            $val = now()->format('d/m/Y');
                            break;
                        case 'hora':
                            $val = now()->format('H:i');
                            break;
                        case 'mesa':
                            $val = $data->mesa ? $data->mesa->name : 'N/A';
                            break;
                        case 'cajero':
                        case 'mesero':
                        case 'usuario':
                            $val = $data->user ? $data->user->name : 'Admin';
                            break;
                        case 'cliente':
                            $val = $data->cliente ?? 'Final';
                            break;
                        case 'total':
                            $val = "RD$ " . number_format($data->total, 2);
                            break;
                    }
                    $buffer .= $label . $val . "\n";
                    break;

                case 'items_table':
                    $buffer .= $LEFT . "Cant  Descripcion        Total\n";
                    $buffer .= "--------------------------------\n";
                    $items = isset($data->detalles) ? $data->detalles : (isset($data->details) ? $data->details : []);

                    foreach ($items as $dp) {
                        $qty = $dp->cantidad ?? $dp->quantity ?? 0;
                        $cant = str_pad($qty, 5, ' ', STR_PAD_RIGHT);
                        $nameLine = substr($dp->product->name ?? 'Prd', 0, 18);
                        $name = str_pad($nameLine, 18, ' ', STR_PAD_RIGHT);

                        // Price without 18% ITBIS
                        $itemSubtotal = $dp->subtotal ?? 0;
                        $totalSinItbis = $itemSubtotal / 1.18;
                        $total = str_pad(number_format($totalSinItbis, 0), 8, ' ', STR_PAD_LEFT);

                        $buffer .= "{$cant}{$name}{$total}\n";

                        // If there are notes, print them below the item
                        if (!empty($dp->observaciones)) {
                            $buffer .= "  * " . $dp->observaciones . "\n";
                        }
                    }
                    break;

                case 'totals':
                    $buffer .= $RIGHT;
                    // Calculation assuming total includes 18% ITBIS
                    $subtotal = $data->total / 1.18;
                    $itbis = $data->total - $subtotal;

                    $buffer .= str_pad("SUBTOTAL:", 15) . str_pad(number_format($subtotal, 2), 12, ' ', STR_PAD_LEFT) . "\n";
                    $buffer .= str_pad("ITBIS (18.00%):", 15) . str_pad(number_format($itbis, 2), 12, ' ', STR_PAD_LEFT) . "\n";
                    $buffer .= $BOLD_ON . str_pad("TOTAL RD$:", 15) . str_pad(number_format($data->total, 2), 12, ' ', STR_PAD_LEFT) . "\n" . $BOLD_OFF;
                    break;

                case 'qr_code':
                    $buffer .= $CENTER . "[ QR CODE ]\n";
                    break;
            }
        }

        $buffer .= "\n\n\n" . $CUT;

        return $buffer;
    }

    private function isBlockActive($struct, $blockId)
    {
        if (!isset($struct['bloques']))
            return true;
        foreach ($struct['bloques'] as $block) {
            if ($block['id'] == $blockId)
                return $block['active'];
        }
        return true;
    }

    /**
     * Send byte buffer to printer socket
     */
    private function sendToPrinter(Impresora $printer, $content)
    {
        if ($printer->tipo === 'network') {
            try {
                $socket = fsockopen($printer->ip, $printer->puerto, $errno, $errstr, 1);
                if (!$socket) {
                    Log::error("Printer connection error: $errstr ($errno)");
                    return false;
                }
                fwrite($socket, $content);
                fclose($socket);
                return true;
            } catch (\Exception $e) {
                Log::error("Printing failed: " . $e->getMessage());
                return false;
            }
        }

        if ($printer->tipo === 'windows' || $printer->tipo === 'usb') {
            try {
                // If the user provided an IP/Path in the 'ip' field (e.g. \\Computer\Printer)
                $path = $printer->ip ?: $printer->nombre;

                // On Windows, you can often write directly to shared printer paths
                // if the web server has permissions.
                file_put_contents($path, $content);
                return true;
            } catch (\Exception $e) {
                Log::error("Direct path printing failed: " . $e->getMessage());
                // Fallback: Just return true to simulate if it's a dev environment
                return false;
            }
        }

        // For simulation or other cases
        Log::info("Print job processed for {$printer->nombre} ({$printer->tipo})");
        return true;
    }
}
