<?php

namespace App\Http\Controllers;

use App\Models\ActivityLog;
use App\Models\AperturaCaja;
use App\Models\Mesa;
use App\Models\Orden;
use App\Models\OrdenDetalle;
use App\Models\Product;
use App\Models\Sale;
use App\Models\SaleDetail;
use App\Models\Impresora;
use App\Services\PrintService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class OrderController extends Controller
{
    public function addItem(Request $request, Orden $orden)
    {
        $start = microtime(true);
        Log::info("addItem Start [Order: {$orden->id}] [" . date('H:i:s.u') . "]");

        $request->validate([
            'product_id' => 'required', // Redundant exists check removed for speed
            'quantity' => 'required|integer|min:1',
        ]);
        Log::info(" - Validation Done: " . (microtime(true) - $start) . "s");

        $product = Product::findOrFail($request->product_id);
        Log::info(" - Product Found: " . (microtime(true) - $start) . "s");

        $existingDetail = $orden->detalles()
            ->where('product_id', $product->id)
            ->where('estado', 'activo')
            ->first();
        Log::info(" - Existing Check Done: " . (microtime(true) - $start) . "s");

        DB::transaction(function () use ($orden, $product, $request, $existingDetail, $start) {
            if ($existingDetail) {
                $existingDetail->increment('cantidad', $request->quantity);
                $existingDetail->update(['subtotal' => $existingDetail->cantidad * $existingDetail->precio_unitario]);
            } else {
                $orden->detalles()->create([
                    'product_id' => $product->id,
                    'cantidad' => $request->quantity,
                    'precio_unitario' => $product->sale_price,
                    'subtotal' => $product->sale_price * $request->quantity,
                    'observaciones' => $request->notes,
                ]);
            }
            Log::info(" - Record Saved/Updated: " . (microtime(true) - $start) . "s");

            if ($orden->estado === 'abierta') {
                $orden->update(['estado' => 'activa']);
                $orden->mesa->update(['status' => 'ocupada']);
            }

            // Optimized total update (Sum only active items)
            $total = $orden->detalles()->where('estado', 'activo')->sum('subtotal');
            $orden->update(['total' => $total, 'subtotal' => $total]);
            Log::info(" - Totals Updated: " . (microtime(true) - $start) . "s");
        });

        Log::info(" - Transaction Committed: " . (microtime(true) - $start) . "s");
        Log::info(" - Wants JSON: " . ($request->wantsJson() ? 'YES' : 'NO'));

        if ($request->ajax() || $request->wantsJson()) {
            Log::info(" - Loading Details...");
            $orden->load('detalles.product');
            Log::info(" - Details Loaded: " . (microtime(true) - $start) . "s");

            $response = response()->json([
                'success' => true,
                'message' => 'Producto agregado.',
                'total' => $orden->total,
                'items' => $orden->detalles->where('estado', 'activo')->map(function ($d) {
                    return [
                        'id' => $d->id,
                        'name' => $d->product->name,
                        'cantidad' => $d->cantidad,
                        'precio_unitario' => (float) $d->precio_unitario,
                        'subtotal' => (float) $d->subtotal,
                        'observaciones' => $d->observaciones
                    ];
                })->values()
            ]);
            Log::info("addItem End (JSON) [Total Time: " . (microtime(true) - $start) . "s]");
            return $response;
        }

        return back()->with('success', 'Producto agregado.');
    }

    public function removeItem(Request $request, OrdenDetalle $detalle)
    {
        if (!Auth::user()->isAdmin()) {
            return $request->ajax() || $request->wantsJson()
                ? response()->json(['success' => false, 'message' => 'Solo administradores pueden eliminar productos.'], 403)
                : back()->with('error', 'Solo administradores pueden eliminar productos.');
        }

        $orden = $detalle->orden;

        DB::transaction(function () use ($detalle, $orden) {
            if ($detalle->cantidad > 1) {
                $detalle->decrement('cantidad');
                $detalle->update(['subtotal' => $detalle->cantidad * $detalle->precio_unitario]);
            } else {
                $detalle->delete();
            }

            $this->updateOrderTotals($orden);

            // If no more active details, revert table/order status
            if ($orden->detalles()->where('estado', 'activo')->count() === 0) {
                $orden->update(['estado' => 'abierta']);
                $orden->mesa->update(['status' => 'libre']);
            }
        });

        if ($request->ajax() || $request->wantsJson()) {
            $orden->load('detalles.product');
            return response()->json([
                'success' => true,
                'message' => 'Producto eliminado.',
                'total' => $orden->total,
                'items' => $orden->detalles->where('estado', 'activo')->map(function ($d) {
                    return [
                        'id' => $d->id,
                        'name' => $d->product->name,
                        'cantidad' => $d->cantidad,
                        'precio_unitario' => (float) $d->precio_unitario,
                        'subtotal' => (float) $d->subtotal,
                        'observaciones' => $d->observaciones
                    ];
                })->values()
            ]);
        }

        return back()->with('success', 'Producto eliminado.');
    }

    public function closeTable(Request $request, Orden $orden, PrintService $printService)
    {
        $start = microtime(true);
        Log::info("closeTable Start [Order: {$orden->id}] [" . date('H:i:s.u') . "]");

        // Check for open cash register (Apertura)
        $caja = \App\Models\Caja::where('is_active', true)->first();
        $cajaAbierta = ($caja && $caja->currentApertura) ? $caja->currentApertura : null;

        Log::info(" - Caja Check: " . (microtime(true) - $start) . "s");

        if (!$cajaAbierta) {
            return back()->with('error', 'No existe una caja abierta en el sistema. Debe abrir caja primero.');
        }

        if ($orden->estado !== 'activa') {
            return back()->with('error', 'La orden no tiene productos confirmados o ya está cerrada.');
        }

        if ($orden->total <= 0) {
            return back()->with('error', 'El total de la orden debe ser mayor a 0.');
        }

        try {
            DB::beginTransaction();

            // 1. Eager Load for stock processing (Reduces DB roundtrips)
            $orden->load('detalles.product.packageItems.product');
            Log::info(" - Details Eager Loaded: " . (microtime(true) - $start) . "s");

            // 2. Create Sale Record (Financial)
            // Calculate remaining amount to account for in this final sale
            // The Final Sale should represent the FINAL transaction amount, but link to ALL items?
            // If we link to ALL items, we deduct ALL stock. Correct.
            // But the Sale Total will be (Total - Accumulated).

            $remainingTotal = $orden->total - $orden->accumulated_paid;
            $discount = 0;

            if ($request->has('discount') && Auth::user()->isAdmin()) {
                $discount = (float) $request->discount;
                if ($discount > $remainingTotal) {
                    throw new \Exception("El descuento no puede ser mayor al balance pendiente.");
                }
                $remainingTotal -= $discount;
            }

            // Validate that payment covers remaining
            if (($request->amount_paid ?? $remainingTotal) < $remainingTotal) {
                throw new \Exception("El monto pagado es menor al restante pendiente (" . number_format($remainingTotal, 2) . ")");
            }

            $sale = Sale::create([
                'apertura_caja_id' => $cajaAbierta->id,
                'user_id' => Auth::id(),
                'total' => $remainingTotal, // Record revenue only for the remaining part
                'discount' => $discount,
                'payment_method' => $request->payment_method ?? 'cash',
                'amount_paid' => $request->amount_paid ?? $remainingTotal,
                'change_given' => ($request->amount_paid ?? $remainingTotal) - $remainingTotal,
                'orden_id' => $orden->id,
                'is_internal' => $orden->mesa->is_internal ? true : false,
            ]);
            Log::info(" - Sale Record Created (Final Close): " . (microtime(true) - $start) . "s");

            // 3. Move items to Sales Details & Deduct Stock
            foreach ($orden->detalles as $detalle) {
                SaleDetail::create([
                    'sale_id' => $sale->id,
                    'product_id' => $detalle->product_id,
                    'quantity' => $detalle->cantidad,
                    'price' => $detalle->precio_unitario, // schema uses 'price'
                    'subtotal' => $detalle->subtotal,
                ]);

                // Deduct Stock
                if ($detalle->product->isBundle()) {
                    // Logic for bundles: Deduct from individual items
                    foreach ($detalle->product->packageItems as $item) {
                        $item->product->decrement('stock', $item->quantity * $detalle->cantidad);
                    }
                } else {
                    // Logic for simple products
                    $detalle->product->decrement('stock', $detalle->cantidad);
                }
            }
            Log::info(" - Details and Stock Processed: " . (microtime(true) - $start) . "s");

            // 4. Close Order & Free Table
            $orden->update([
                'estado' => 'facturada',
                'fecha_cierre' => now(),
                'pre_cuenta_generada' => false,
            ]);

            $orden->mesa->update(['status' => 'libre']);
            Log::info(" - Order Status/Mesa Freed: " . (microtime(true) - $start) . "s");

            // 5. Log
            ActivityLog::create([
                'user_id' => Auth::id(),
                'action' => 'Cierre de Mesa',
                'description' => "Mesa {$orden->mesa->name} cerrada. Venta #{$sale->id} generada.",
            ]);

            DB::commit();
            Log::info("closeTable Committed: " . (microtime(true) - $start) . "s");

            // Printing (Outside transaction for performance)
            try {
                $printService->printOrder($orden, 'factura');
            } catch (\Exception $e) {
                Log::error("Deferred printing failed: " . $e->getMessage());
            }

            $responseData = ['success' => 'Mesa cerrada y venta registrada correctamente.'];
            $defaultPrinter = Impresora::where('es_predeterminada', true)->first();

            if ($defaultPrinter && $defaultPrinter->use_browser) {
                $responseData['print_order'] = $orden->id;
                $responseData['print_type'] = 'factura';
            }

            return redirect()->route('pos.map')->with($responseData);

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error al cerrar mesa: ' . $e->getMessage());
        }
    }

    public function splitPayment(Request $request, Orden $orden, PrintService $printService)
    {
        $request->validate([
            'items' => 'required|array|min:1',
            'items.*.id' => 'required|exists:orden_detalles,id',
            'items.*.quantity' => 'required|integer|min:1',
            'payment_method' => 'required|in:cash,card,transfer',
            'amount_paid' => 'nullable|numeric|min:0',
        ]);

        $caja = \App\Models\Caja::where('is_active', true)->first();
        $apertura = ($caja && $caja->currentApertura) ? $caja->currentApertura : null;

        if (!$apertura) {
            return response()->json(['success' => false, 'message' => 'No hay una caja abierta.'], 403);
        }

        try {
            DB::beginTransaction();

            $splitTotal = 0;
            $saleItems = [];

            foreach ($request->items as $splitItem) {
                $detalle = OrdenDetalle::where('orden_id', $orden->id)
                    ->lockForUpdate()
                    ->findOrFail($splitItem['id']);

                if ($detalle->cantidad < $splitItem['quantity']) {
                    throw new \Exception("Cantidad insuficiente para el producto: {$detalle->product->name}");
                }

                $itemSubtotal = $detalle->precio_unitario * $splitItem['quantity'];
                $splitTotal += $itemSubtotal;

                $saleItems[] = [
                    'product_id' => $detalle->product_id,
                    'quantity' => $splitItem['quantity'],
                    'price' => $detalle->precio_unitario,
                    'subtotal' => $itemSubtotal,
                    'detalle_id' => $detalle->id
                ];

                // Deduct from original detail
                $detalle->decrement('cantidad', $splitItem['quantity']);
                $detalle->update(['subtotal' => $detalle->cantidad * $detalle->precio_unitario]);

                if ($detalle->cantidad <= 0) {
                    $detalle->delete(); // Or mark as paid
                }
            }

            // Create Sale Record
            $sale = Sale::create([
                'apertura_caja_id' => $apertura->id,
                'user_id' => Auth::id(),
                'total' => $splitTotal,
                'payment_method' => $request->payment_method,
                'amount_paid' => $request->amount_paid ?? $splitTotal,
                'change_given' => max(0, ($request->amount_paid ?? $splitTotal) - $splitTotal),
                'orden_id' => $orden->id,
                'is_internal' => $orden->mesa->is_internal ? true : false,
            ]);

            foreach ($saleItems as $sItem) {
                SaleDetail::create([
                    'sale_id' => $sale->id,
                    'product_id' => $sItem['product_id'],
                    'quantity' => $sItem['quantity'],
                    'price' => $sItem['price'],
                    'subtotal' => $sItem['subtotal'],
                ]);

                // Deduct Stock
                Product::where('id', $sItem['product_id'])->decrement('stock', $sItem['quantity']);
            }

            // Update original order totals
            $this->updateOrderTotals($orden);

            // If no more items, close the order
            if ($orden->detalles()->where('estado', 'activo')->count() === 0) {
                $orden->update([
                    'estado' => 'facturada',
                    'fecha_cierre' => now(),
                    'pre_cuenta_generada' => false,
                ]);
                $orden->mesa->update(['status' => 'libre']);
                $isOrderClosed = true;
            } else {
                $isOrderClosed = false;
            }

            ActivityLog::create([
                'user_id' => Auth::id(),
                'action' => 'Pago Parcial',
                'description' => "Mesa {$orden->mesa->name}. Venta Parcial #{$sale->id} generada.",
            ]);

            DB::commit();

            // Trigger silent printing
            try {
                $printService->printSale($sale);
            } catch (\Exception $e) {
                Log::error("Partial payment printing failed: " . $e->getMessage());
            }

            $responseData = [
                'success' => true,
                'message' => 'Pago parcial registrado correctamente.',
                'is_closed' => $isOrderClosed,
                'redirect' => $isOrderClosed ? route('pos.map') : null
            ];

            $defaultPrinter = \App\Models\Impresora::where('es_predeterminada', true)->first();
            if ($defaultPrinter && $defaultPrinter->use_browser) {
                $responseData['print_order'] = $orden->id;
                $responseData['print_type'] = 'factura';
            }

            return response()->json($responseData);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function payPartialAmount(Request $request, Orden $orden, PrintService $printService)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1',
            'payment_method' => 'required|in:cash,card,transfer',
            'amount_paid' => 'nullable|numeric|min:0',
        ]);

        $caja = \App\Models\Caja::where('is_active', true)->first();
        $apertura = ($caja && $caja->currentApertura) ? $caja->currentApertura : null;

        if (!$apertura) {
            return response()->json(['success' => false, 'message' => 'No hay una caja abierta.'], 403);
        }

        // Validate amount against remaining balance
        $currentPaid = $orden->accumulated_paid;
        $remaining = $orden->total - $currentPaid;

        if ($request->amount > $remaining) {
            return response()->json(['success' => false, 'message' => 'El monto excede el balance pendiente.'], 400);
        }

        try {
            DB::beginTransaction();

            $payAmount = $request->amount;
            $discount = 0;

            if ($request->has('discount') && Auth::user()->isAdmin()) {
                $discount = (float) $request->discount;
                if ($discount > $payAmount) {
                    throw new \Exception("El descuento no puede ser mayor al monto a pagar.");
                }
                $payAmount -= $discount;
            }

            $received = $request->amount_paid ?? $payAmount;
            $change = max(0, $received - $payAmount);

            // Create Partial Sale Record (Revenue only, no stock)
            $sale = Sale::create([
                'apertura_caja_id' => $apertura->id,
                'user_id' => Auth::id(),
                'total' => $payAmount,
                'discount' => $discount,
                'payment_method' => $request->payment_method,
                'amount_paid' => $received,
                'change_given' => $change,
                'orden_id' => $orden->id,
                'is_internal' => $orden->mesa->is_internal ? true : false,
            ]);

            // Update Orden with full amount including discount (since the debt is cleared)
            $orden->increment('accumulated_paid', $request->amount);

            ActivityLog::create([
                'user_id' => Auth::id(),
                'action' => 'Pago Parcial (Monto)',
                'description' => "Mesa {$orden->mesa->name}. Pago Parcial #{$sale->id} de RD$ " . number_format($payAmount, 2),
            ]);

            DB::commit();

            // Trigger silent printing
            try {
                $printService->printSale($sale);
            } catch (\Exception $e) {
                Log::error("Partial payment printing failed: " . $e->getMessage());
            }

            // Check if fully paid (tolerant float comparison)
            $newRemaining = $orden->total - $orden->accumulated_paid;
            $isFullyPaid = abs($newRemaining) < 0.01;

            $responseData = [
                'success' => true,
                'message' => 'Pago parcial registrado correctamente.',
                'remaining' => $newRemaining,
                'redirect' => null
            ];

            return response()->json($responseData);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false, 'message' => $e->getMessage()], 500);
        }
    }

    public function printTicket(Request $request, Orden $orden)
    {
        $orden->load(['detalles.product', 'mesa', 'user']);
        $tipo = $request->get('tipo', 'precuenta');
        $sale = null;
        if ($orden->estado === 'facturada') {
            $sale = Sale::where('orden_id', $orden->id)->latest()->first();
        }

        $config = \App\Models\BusinessConfig::first() ?: new \App\Models\BusinessConfig([
            'business_name' => 'SUKHA CIGAR BAR',
            'rnc' => null,
            'phone' => null,
            'address' => null
        ]);

        return view('pos.print-ticket', compact('orden', 'tipo', 'sale', 'config'));
    }

    public function printPreCheck(Orden $orden, PrintService $printService)
    {
        if ($orden->detalles->isEmpty()) {
            return back()->with('error', 'No hay productos para generar pre-cuenta.');
        }

        $orden->update(['pre_cuenta_generada' => true]);

        // Printing (Outside main flow for performance)
        try {
            $printService->printOrder($orden, 'precuenta');
        } catch (\Exception $e) {
            Log::error("Precheck printing failed: " . $e->getMessage());
        }

        $responseData = ['success' => 'Pre-cuenta generada correctamente.'];
        $defaultPrinter = Impresora::where('es_predeterminada', true)->first();

        if ($defaultPrinter && $defaultPrinter->use_browser) {
            $responseData['print_order'] = $orden->id;
            $responseData['print_type'] = 'precuenta';
        }

        return back()->with($responseData);
    }

    public function cancel(Orden $orden)
    {
        if (!Auth::user()->isAdmin()) {
            return back()->with('error', 'Solo administradores pueden anular ventas/órdenes.');
        }

        if ($orden->estado === 'facturada') {
            return back()->with('error', 'No se puede cancelar una orden ya facturada.');
        }

        DB::transaction(function () use ($orden) {
            $orden->detalles()->delete();
            $orden->mesa->update(['status' => 'libre']);
            $orden->delete();
        });

        return redirect()->route('pos.map')->with('success', 'Orden cancelada y mesa liberada.');
    }

    public function updateCliente(Request $request, Orden $orden)
    {
        $request->validate([
            'cliente' => 'nullable|string|max:255',
        ]);

        $orden->update(['cliente' => $request->cliente]);

        return response()->json([
            'success' => true,
            'cliente' => $orden->cliente,
        ]);
    }

    private function updateOrderTotals(Orden $orden)
    {
        $subtotal = $orden->detalles()->where('estado', 'activo')->sum('subtotal');
        // Tax logic can be added here
        $tax = 0;
        $total = $subtotal + $tax;

        $orden->update([
            'subtotal' => $subtotal,
            'impuesto' => $tax,
            'total' => $total,
        ]);
    }
}
