<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Caja;
use App\Models\AperturaCaja;
use App\Models\Arqueo;
use App\Models\Orden;
use App\Models\ActivityLog;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class CajaController extends Controller
{
    // Show status or redirect to open/close
    public function index()
    {
        // Assuming a single cashier works on a single register for now, or selecting one.
        // For simplicity, we'll pick the first active caja.
        $caja = Caja::where('is_active', true)->first();

        if (!$caja) {
            return "No hay cajas configuradas.";
        }

        $apertura = $caja->currentApertura;

        if ($apertura) {
            return view('caja.status', compact('caja', 'apertura'));
        }

        return view('caja.open', compact('caja'));
    }

    public function open(Request $request)
    {
        $request->validate([
            'caja_id' => 'required|exists:cajas,id',
            'monto_inicial' => 'required|numeric|min:0',
            'fecha_inicio' => 'required|date',
            'fecha_fin' => 'required|date|after:fecha_inicio',
        ]);

        $caja = Caja::findOrFail($request->caja_id);

        if ($caja->currentApertura) {
            return redirect()->back()->withErrors(['msg' => 'La caja ya está abierta.']);
        }

        $apertura = AperturaCaja::create([
            'caja_id' => $caja->id,
            'user_id' => Auth::id(),
            'monto_inicial' => $request->monto_inicial,
            'fecha_apertura' => Carbon::now(),
            'fecha_inicio' => $request->fecha_inicio,
            'fecha_fin' => $request->fecha_fin,
            'estado' => 'abierta',
        ]);

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'open_box',
            'description' => "Caja {$caja->name} abierta con {$request->monto_inicial}",
            'ip_address' => $request->ip(),
        ]);

        return redirect()->route('caja.operations');
    }

    public function showCloseForm()
    {
        $caja = Caja::where('is_active', true)->first();
        $apertura = $caja->currentApertura;

        if (!$apertura) {
            return redirect()->route('caja.operations');
        }

        $openTablesCount = Orden::whereIn('estado', ['abierta', 'activa'])
            ->has('detalles')
            ->count();

        return view('caja.close', compact('caja', 'apertura', 'openTablesCount'));
    }

    public function close(Request $request, \App\Services\PrintService $printService)
    {
        $request->validate([
            'apertura_id' => 'required|exists:apertura_cajas,id',
            'monto_fisico' => 'required|numeric|min:0',
            'denominaciones' => 'nullable|array',
        ]);

        $apertura = AperturaCaja::findOrFail($request->apertura_id);

        // Automatically clean up orders with no items
        Orden::whereIn('estado', ['abierta', 'activa'])
            ->doesntHave('detalles')
            ->delete();

        // Note: The warning is now handled in the frontend via a modal
        // We allow the closure even if there are open orders as requested

        // Calculate system amount (Initial + Sales Cash - Expenses)
        // Internal sales (Consumption) do not affect cash flow
        $totalVentasEfectivo = $apertura->sales()
            ->where('payment_method', 'cash')
            ->where('is_internal', false)
            ->sum('total');

        $totalVentasTarjeta = $apertura->sales()
            ->where('payment_method', 'card')
            ->where('is_internal', false)
            ->sum('total');

        $totalVentasTransferencia = $apertura->sales()
            ->where('payment_method', 'transfer')
            ->where('is_internal', false)
            ->sum('total');

        $totalConsumoInterno = $apertura->sales()
            ->where('is_internal', true)
            ->sum('total');

        // Future: Subtract expenses if implemented
        $monto_sistema = $apertura->monto_inicial + $totalVentasEfectivo;

        $diferencia = $request->monto_fisico - $monto_sistema;

        $arqueo = Arqueo::create([
            'apertura_caja_id' => $apertura->id,
            'user_id' => Auth::id(),
            'monto_sistema' => $monto_sistema,
            'monto_fisico' => $request->monto_fisico,
            'monto_tarjeta' => $totalVentasTarjeta,
            'monto_transferencia' => $totalVentasTransferencia,
            'denominaciones' => $request->denominaciones,
            'diferencia' => $diferencia,
            'observaciones' => $request->observaciones,
        ]);

        $apertura->update([
            'monto_final' => $request->monto_fisico,
            'fecha_cierre' => Carbon::now(),
            'estado' => 'cerrada',
        ]);

        ActivityLog::create([
            'user_id' => Auth::id(),
            'action' => 'close_box',
            'description' => "Caja cerrada. Diferencia: {$diferencia}",
            'ip_address' => $request->ip(),
        ]);

        // Print the Arqueo report automatically
        try {
            $printService->printArqueo($arqueo);
        } catch (\Exception $e) {
            Log::error("Arqueo printing failed: " . $e->getMessage());
        }

        return redirect()->route('caja.summary', $arqueo->id)->with('success', 'Caja cerrada correctamente.');
    }

    public function summary(Arqueo $arqueo)
    {
        $arqueo->load('apertura.sales', 'user');
        $apertura = $arqueo->apertura;

        return view('caja.summary', compact('arqueo', 'apertura'));
    }
}
